/// @file deviceDetector.h
///
/// @brief Device detection main header
///
/// This file defines the DeviceDetector class and the enum it uses.
/// The aim of this class is to provide a common interface for USPI components
/// to detect devices' events and get information from them.
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see deviceManagement.h deviceDetector.cpp Udev

#ifndef __DEVICE_DETECTOR_H__
#define __DEVICE_DETECTOR_H__

#include "deviceManagement.h"

namespace adit {
namespace uspi {

/// @enum DD_LOG_TYPE
/// @brief DeviceDetector log type.
///
/// It defines the kind of output, if any, that must be used by the device
/// detection library.
enum DD_LOG_TYPE {
    DD_LOG_NONE = 0,            ///< No output
    DD_LOG_DLT,                 ///< Output to Diagnosis %Log and Trace
    DD_LOG_SYSLOG,              ///< Output to System %Log
    DD_LOG_STDOUT,              ///< Output to standard out
    DD_LOG_COUT = DD_LOG_STDOUT,///< Output to standard out
    DD_LOG_CLOG                 ///< Output to standard log
};

/// @enum DD_EVENT_TYPE
/// @brief DeviceDetector reported event types.
///
/// Lists the different kind of events that will be reported by the device
/// detection class.
enum DD_EVENT_TYPE {
    DD_EV_ADD = 0,              ///< New device added
    DD_EV_REMOVE,               ///< Device removed
    DD_EV_CHANGE                ///< Device has changed
};

/// @enum DD_EVENT_MASK
/// @brief DeviceDetector events to watch.
///
/// The device detection is capable of simple filtering for devices to watch.
/// This list may be extended to support WIFI devices, and other subsets of the
/// current supported ones.
enum DD_EVENT_MASK {
    DD_USB_OTHER                = 1 << 0, ///< Any USB that is not listed below
    DD_USB_MASS_STORAGE         = 1 << 1, ///< USB mass storage
    DD_USB_UNWIRED_TECHNOLOGY   = 1 << 2, ///< USB Unwired HUB
    DD_USB_HUB                  = 1 << 3, ///< USB HUB
    DD_USB_APPLE                = 1 << 4, ///< USB Apple device
    /// Helper for AOAP, includes DD_USB_OTHER and DD_USB_MASS_STORAGE
    DD_USB_AOAP                 = (DD_USB_OTHER | DD_USB_MASS_STORAGE),
    DD_BLUETOOTH_OTHER          = 1 << 16, ///< Any Bluetooth device
    DD_ALL                      = 0xffffffff ///< All the possible masks
};

class IDeviceEventHandler {
public:
    /// @brief Called by @ref dispatchEvent() or the EventHandler
    /// @param devInfo The device information
    /// @param eventType The event type to be considered
    /// @param eventMask The device mask that has been triggered
    ///
    /// Must be implemented by the final user. This method is executed each time
    /// a device matching the @ref mMask changes its state.
    virtual void handleDeviceEvent(DeviceInfo& devInfo,
                                   int eventType,
                                   int eventMask) = 0;
};

/// @brief The device detection class.
///
/// This class is meant to be derived by the user. It provides basic
/// functionalities to detect devices' events for those matching a specific
/// mask. By default, this object provides the file descriptor to listen the
/// events to, and the method to dispatch the received events.
/// The dispatcher will execute the user defined @ref handleDeviceEvent().
/// The methods of this object are **not thread safe**.
class DeviceDetector {
    /// The device mask to watch for.
    uint32_t mMask;
    /// Whether the object is in standalone mode
    bool mStandAlone;

    /// The device data.
    /// This @ref DeviceInfo object is populated with the data retrieved from
    /// the device that generated the latest event. It can be used by the client
    /// to retrieve valuable information.
    DeviceInfo mDevInfo;

    /// Checks if a device must generate a notification.
    /// This method is executed by the Udev class once a new device has been
    /// added or removed from the device list.
    void checkDevice(DeviceInfo &devInfo, enum DD_EVENT_TYPE);
    friend class Udev;

    IDeviceEventHandler& mDeviceHandler;
public:
    // Object constructor
    // logType The type of log to initialize.
    DeviceDetector(IDeviceEventHandler& deviceHandler,
                   enum DD_LOG_TYPE logType = DD_LOG_DLT);
    // Object destructor
    // Releases all the resources and stop the EventHandler thread if started.
    virtual ~DeviceDetector();

    /// @brief Starts the stand alone mode.
    ///
    /// Not thread safe. The EventHandler thread will be created and the object
    /// @ref mStandAlone will be set to true. Once this method is executed, the
    /// @ref getEventFd() and @ref dispatchEvent() throw an error if executed.
    void setStandAlone(void);

    /// Retrieves the file descriptor to be polled on.
    int getEventFd(void);
    /// Dispatches the event occurring on the @ref getEventFd() file descriptor.
    int dispatchEvent(void);

    /// @brief Sets the @ref mMask value.
    /// @param inMask The mask to set.
    ///
    /// This will trigger the check of the device list. Any newly watched device
    /// will generate an event, and thus call to @ref handleDeviceEvent() will
    /// be performed.
    void setMask(uint32_t inMask);
};

} // namespace uspi
} // namespace adit
#endif // __DEVICE_DETECTOR_H__
